package com.swhotels.reservation;

/*
* La classe <code>ReservationService</code> implmente 
* l'interface <code>IReservationService</code> dont l'objectif 
* consiste  offrir un service d'accs au systme de rservation 
* de la centrale de rservation SW-Htels.
*
* @version  1.0
*/
public class ReservationService implements IReservationService {

	/*
	* Le gestionnaire de toutes les rservations locales
	* en cours.
	*/
	private static ReservationManager manager;

	/*
	* Le constructeur du service de rservation.
	*/
	public ReservationService() {
	}

	/*
	* Renvoie le gestionnaire des rservations en cours.
	*
 	* @return		ReservationManager le singleton du
 	*				gestionnaire des rservations en cours.
	*/
	private synchronized ReservationManager getManager() {
		if (manager==null)
			manager=ReservationManager.getUniqueInstance() ;
		return manager;
	}

	/*
	* Recherche de disponibilits pour la rservation d'un
	* htel.
	*
	* @param		passengers le nombre de passagers
	* @param		from la localit de dpart
	* @param		to la localit d'arrive
	* @param		roundTrip aller retour/aller simple
	* @param		departureDay jour de dpart
	* @param		departureMonth mois de dpart 
	* @param		departureYear anne de dpart
	* @param		arrivalDay jour d'arrive
	* @param		arrivalMonth mois d'arrive 
	* @param		arrivalYear anne d'arrive
	* @param		remoteId identifiant de rservation du partenaire
	*				de rservation
	*
 	* @return		int le n de rservation affect  la recherche
 	*				de disponibilits du client
	*/
	public int search(int passengers, String from, String to, boolean roundTrip, 
		int departureDay, int departureMonth, int departureYear, int arrivalDay, 
		int arrivalMonth, int arrivalYear, int remoteId) throws ReservationException {

		// Cre et enregistre une nouvelle rservation
		// dans le registre local de SW-Htels.
		ReservationManager manager = getManager();
		Reservation reservation = new Reservation();

		reservation.setPassengers(passengers);
		reservation.setFrom(from);
		reservation.setTo(to);
		reservation.setRoundTrip(roundTrip);
		reservation.setDepartureDay(departureDay);
		reservation.setDepartureMonth(departureMonth);
		reservation.setDepartureYear(departureYear);
		reservation.setArrivalDay(arrivalDay);
		reservation.setArrivalMonth(arrivalMonth);
		reservation.setArrivalYear(arrivalYear);
		reservation.setRemoteId(remoteId);

		int id = reservation.getId();
		manager.newReservation((Object)new Integer(id), (Object)reservation);
		return id;
	}

	/*
	* Confirmation de la rservation dont l'identifiant est
	* pass en paramtre.
	*
	* @param		reservationId le n de la rservation 
	*				confirmer
	* @param		availabilityId le n de disponibilit 
	*				confirmer
	*
	* @return		<code>true</code> si la confirmation s'est
	*				bien passe; <code>false</code> sinon
	*/
	public boolean book(int reservationId, int availabilityId)
		throws ReservationException {

	    // Recherche de la rservation locale
		Reservation reservation = getReservation(reservationId);

		reservation.status(availabilityId);

		// Confirmation de la rservation locale (ici, la
		// rservation est supprime, mais dans un systme
		// systme plus proche de la ralit, elle serait
		// transmise  un logiciel de back-office).
		ReservationManager manager = getManager();
		manager.removeReservation((Object)new Integer(reservationId));
		return true;
	}

	/*
	* Annulation de la rservation dont l'identifiant est
	* pass en paramtre.
	*
	* @param		reservationId le n de la rservation 
	*				annuler
	*
	* @return		<code>true</code> si l'annulation s'est
	*				bien passe; <code>false</code> sinon
	*/
	public boolean cancel(int reservationId)
		throws ReservationException {

	    // Recherche de la rservation locale.
		Reservation reservation = getReservation(reservationId);

		reservation.status(0);

		// Annulation de la rservation locale.
		ReservationManager manager = getManager();
		manager.removeReservation((Object)new Integer(reservationId));
		return true;
	}

	/*
	* Renvoie la rservation locale dont l'identifiant
	* est pass en paramtre.
	*
	* @param		reservationId l'identifiant de la
	*				rservation locale
	*
 	* @return		Reservation la rservation locale dont
 	*				l'identifiant est pass en paramtre.
	*/
	private Reservation getReservation(int reservationId) {

		ReservationManager manager = getManager();
		Reservation reservation = (Reservation)manager.getReservation((Object)new Integer(reservationId));
		if (reservation == null) { 
			throw new IllegalArgumentException ("la rservation (id="+reservationId+") SW-Htels n'est pas enregistre.");
		}
		return reservation;
	}

	/*
	* Renvoie les disponibilits de la centrale de rservation 
	* htelire SW-Htels pour la rservation locale dont 
	* l'identifiant est pass en paramtre.
	*
	* @param		reservationId l'identifiant de la
	*				rservation locale
	*
 	* @return		HotelAvailability[] les disponibilits de la
 	*				centrale de rservation htelire pour la 
 	*				rservation locale dont l'identifiant est 
 	*				pass en paramtre.
	*/
	public HotelAvailability[] getHotelAvailabilities(int reservationId) {

	    // Recherche de la rservation locale
		Reservation reservation = getReservation(reservationId);

		return reservation.getHotelAvailabilities();
	}
}